<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class DatabaseBackup extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:backup';


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Backup the database every hour';


    /**
     * Execute the console command.
    */
    public function handle()
    {
        // Get credentials from .env and trim
        $database = trim(config('database.connections.mysql.database'));
        $username = trim(config('database.connections.mysql.username'));
        $password = trim(config('database.connections.mysql.password'));
        $host     = trim(config('database.connections.mysql.host'));

        // Validate credentials
        if (!$database || !$username || !$password || !$host) {
            $this->error('Database credentials are not set correctly.');
            return 1;
        }

        // Create backup directory if needed
        $backupDir = storage_path('app/backup');
        if (!Storage::exists('backup')) {
            Storage::makeDirectory('backup');
        }

        // Define backup file
        $fileName = 'backup-' . Carbon::now()->format('Y-m-d_H-i-s') . '.sql';
        $filePath = "{$backupDir}/{$fileName}";

        // Run mysqldump
        $command = "mysqldump --user=\"{$username}\" --password=\"{$password}\" --host=\"{$host}\" \"{$database}\" > \"{$filePath}\"";
        $output = null;
        $resultCode = null;
        exec($command, $output, $resultCode);

        // Check if backup was successful and file is not empty
        if ($resultCode === 0 && file_exists($filePath) && filesize($filePath) > 100) {
            $this->info("✅ Backup created: {$fileName}");

            // Delete backups older than 2 day
            $files = Storage::files('backup');
            foreach ($files as $file) {
                if (Storage::lastModified($file) < now()->subDay(1)->timestamp && $file !== "backup/{$fileName}") {
                    Storage::delete($file);
                    $this->info("🗑️ Deleted old backup: {$file}");
                }
            }

        } else {
            $this->error("❌ Backup failed or file is empty!");

            if (file_exists($filePath)) {
                unlink($filePath);
                $this->warn("🧹 Removed empty backup file: {$fileName}");
            }
        }

        return $resultCode;
    }
}
